unit OpcodeLookup;

interface

uses
  Classes, SysUtils, IniFiles;

type
  TParamType = (ptInteger, ptFloat, ptLabel, ptGlobalVar, ptLocalVar,
    ptObjectRef, ptMissionRef, ptString);

  TOpCodeDef = class
    Code: Word;
    Name: string;
    NumParams: Integer;
    Params: set of TParamType;
  end;

  TOpcodeLookup = class
  protected
    FByCode: TList;
    FByName: TList;
  public
    constructor Create;
    destructor Destroy; override;
    procedure Load(OpDef: TFileName);
    procedure Clear;
    function GetOpByName(Name: string; Len: Integer): TOpCodeDef;
  end;


implementation

function CompareNames(Item1, Item2: Pointer): Integer;
begin
  Result := CompareText(TOpCodeDef(Item1).Name, TOpCodeDef(Item2).Name);
end;

function CompareCode(Item1, Item2: Pointer): Integer;
begin
  Result := TOpCodeDef(Item1).Code - TOpCodeDef(Item2).Code;
end;


{ TOpcodeLookup }

constructor TOpcodeLookup.Create;
begin
  FByCode := TList.Create;
  FByName := TList.Create;
end;

destructor TOpcodeLookup.Destroy;
begin
  Clear;
  FByCode.Free;
  FByName.Free;
end;

procedure TOpcodeLookup.Load(OpDef: TFileName);
var
  Ini: TMemIniFile;
  i: Integer;
  s: string;
  op: TOpCodeDef;
begin
  Ini := TMemIniFile.Create(OpDef);

  for i := 0 to 1999 do
  begin
    s := Ini.ReadString('opcodes', IntToHex(i, 4), '');
    if s <> '' then
    begin
      op := TOpCodeDef.Create;
      op.Code := i;
      op.NumParams := StrToInt(copy(s,1, Pos(',', s)-1));
      op.Name := copy(s, Pos(',',s)+1, Length(s)- Pos(',',s));

      // Only add opcode to name list if it has a name.
      if Op.Name <> '' then
        FByName.Add(op);

      FByCode.Add(op);
    end;
  end;
  Ini.Free;

  FByName.Sort(@CompareNames);
  // Already sorted by code in the def file but you can never be too safe
  FByCode.Sort(@CompareCode);
end;

procedure TOpcodeLookup.Clear;
var
  i: Integer;
begin
  for i := 0 to FByCode.Count -1 do
  begin
    TOpCodeDef(FByCode.List[i]).Free;
  end;
  FByCode.Clear;
  FByName.Clear;
end;

function TOpcodeLookup.GetOpByName(Name: string; Len: Integer): TOpCodeDef;
var
  first, mid, last, i,n: Integer;
begin
  first := 0;
  last := FByName.Count - 1;
  Result := nil;
  while first <= Last do
  begin
    mid := (first + last) div 2;
    i := CompareText(Name, TOpCodeDef(FByName.Items[mid]).Name);
    if i = 0 then
    begin
      Result := TOpCodeDef(FByName.Items[mid]);
      Exit;
    end;
    if i < 0 then
      last := mid -1
    else
      first := mid + 1;
  end;

  try
    if Len = 4 then
    begin
      first := 0;
      last := FByCode.Count - 1;
      n := StrToInt('$' + Copy(Name, 1, 4));

      while first <= Last do
      begin
        mid := (first + last) div 2;
        if TOpCodeDef( FByCode.Items[mid] ).Code = n then
        begin
          Result := TOpCodeDef(FByCode.Items[mid]);
          Exit;
        end;
        if n < TOpCodeDef( FByCode.Items[mid] ).Code then
          last := mid -1
        else
          first := mid + 1;
      end;
    end;
  except
  end;
end;

end.

